package gov.va.vinci.dart.common;

import gov.va.vinci.dart.common.exception.ValidationException;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

public class ValidationHelper {
	public static final SimpleDateFormat SDF_DATE = new SimpleDateFormat(
			"MM/dd/yyyy");
	
	public static void required(final String name, final Object value)
			throws ValidationException {
		if (value == null) {
			throw new ValidationException(name + " is required.");
		}
	}

	public static void required(final String name, final String value)
			throws ValidationException {
		if (value == null || value.length() < 1) {
			throw new ValidationException(name + " is required.");
		}
	}

	public static void validateSize(final String name, final String value, int min, int max)
			throws ValidationException {
		if (value == null) {
			if (min > 0) {
				throw new ValidationException(name + " is required.");
			}
			return;
		}

		int len = value.length();

		if (len < min || len > max) {
			throw new ValidationException(name + " must be between " + min
					+ " and " + max + " characters long.");
		}
	}

	public static void validateWholeNumber(final String name, final String value, int max) throws ValidationException  {
		if (value == null ) {
			throw new ValidationException(name + " is required.");
		}

		int len = value.length();
		
		if (len < 1 || len > max) {
			throw new ValidationException(name + " must be a number between 0 and " + max + " digits.");
		}

		if (value.matches("[0-9]+") == false) {
			throw new ValidationException(name + " must be a number.");
		}
	}
	
	public static void validateDateRange(final String name, final Date value, final Date minDate, final Date maxDate) throws ValidationException  {
		if (value == null ) {
			throw new ValidationException(name + " is required.");
		}

		if (value.before(minDate) || value.after(maxDate)) {
			throw new ValidationException(name + " must be between " + SDF_DATE.format(minDate) + " and " + SDF_DATE.format(maxDate));
		}
	}
	
	public static void validateDateBefore(final String name, final Date value, final Date maxDate) throws ValidationException  {
		if (value == null ) {
			throw new ValidationException(name + " is required.");
		}

		if (value.before(maxDate) == false) {
			throw new ValidationException(name + " must be before " + SDF_DATE.format(maxDate));
		}
	}

	public static void validateDateAfter(final String name, final Date value, final Date minDate) throws ValidationException  {
		if (value == null ) {
			throw new ValidationException(name + " is required.");
		}

		if (value.after(minDate) == false) {
			throw new ValidationException(name + " must be after " + SDF_DATE.format(minDate));
		}
	}

	public static Date validateDate(final String name, final String value) throws ValidationException  {
		if (value == null ) {
			throw new ValidationException(name + " is required.");
		}

		try {
			return SDF_DATE.parse(value);
		}
		catch (ParseException e) {
			throw new ValidationException(name + " must be a valid date in the format mm/dd/yyyy.");
		}
	}

	public static void validateNotBothNull(final String name1, final String name2, final Object obj1, final Object obj2) throws ValidationException  {
		if (obj1 == null && obj2 == null) {
			throw new ValidationException(name1 + " and " + name2 + " cannot both be null.");
		}
	}
	
}
